<?php
/**
 * @copyright   Copyright (c) 2021 Urbano Argentina (http://www.urbano.com.ar)
 * @license     http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 * @version     2.2.0.3
 */

namespace Urbano\Shipping\Model\Carrier;

use Magento\Framework\DataObject;
use Magento\Quote\Model\Quote\Address\RateResult\Method;


class Urbanoshipping extends \Magento\Shipping\Model\Carrier\AbstractCarrier implements \Magento\Shipping\Model\Carrier\CarrierInterface
{
    /**
    * Carrier's code
    * @var string
    */
    protected $_code = 'urbano_shipping';

    protected $_scopeConfig;
    
    /**
    * Whether this carrier has fixed rates calculation
    * @var bool
    */
    protected $_isFixed = true;

    /** @var \Magento\Shipping\Model\Rate\ResultFactory */
    protected $_rateResultFactory;

    /** @var \Magento\Quote\Model\Quote\Address\RateResult\MethodFactory */
    protected $_rateMethodFactory;

    /** @var \Psr\Log\LoggerInterface */
    protected $_logger;

    /** @var \Magento\Framework\Message\ManagerInterface */
    protected $_messageManager;

    /** @var \Urbano\Shipping\Helper\Urbano */
    protected $_urbano;

    /** @var \Magento\Shipping\Model\Tracking\Result\StatusFactory */
    protected $_trackStatusFactory;

    /** @var bool */
    private $_enabled_custom_price;

    /** @var int|float */
    private $_shipping_price;

    /**
     * @param \Psr\Log\LoggerInterface $logger
     * @param \Urbano\Shipping\Helper\Urbano $urbano
     * @param \Magento\Shipping\Model\Rate\ResultFactory $rateResultFactory
     * @param \Magento\Framework\Message\ManagerInterface $messageManager
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     * @param \Magento\Shipping\Model\Tracking\Result\StatusFactory $trackStatusFactory
     * @param \Magento\Quote\Model\Quote\Address\RateResult\ErrorFactory $rateErrorFactory
     * @param \Magento\Quote\Model\Quote\Address\RateResult\MethodFactory $rateMethodFactory
     * @param array $data
     */
    public function __construct(
        \Psr\Log\LoggerInterface $logger,
        \Urbano\Shipping\Helper\Urbano $urbano,
        \Magento\Shipping\Model\Rate\ResultFactory $rateResultFactory,
        \Magento\Framework\Message\ManagerInterface $messageManager,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Shipping\Model\Tracking\Result\StatusFactory $trackStatusFactory,
        \Magento\Quote\Model\Quote\Address\RateResult\ErrorFactory $rateErrorFactory,
        \Magento\Quote\Model\Quote\Address\RateResult\MethodFactory $rateMethodFactory,
        array $data = []
    ) {
        $this->_messageManager = $messageManager;
        $this->_rateResultFactory = $rateResultFactory;
        $this->_rateMethodFactory = $rateMethodFactory;
        $this->_rateErrorFactory = $rateErrorFactory;
        $this->_urbano = $urbano;
        $this->_logger = $logger;
        $this->_trackStatusFactory = $trackStatusFactory;
        $this->_scopeConfig = $scopeConfig;

        parent::__construct($scopeConfig, $rateErrorFactory, $logger, $data);
    }

    /**
     * Genera la lista de los metodos de envio aceptados.
     *
     * @return array
     * @api
     */
    public function getAllowedMethods()
    {
        return [$this->getCarrierCode() => __($this->getConfigData('name'))];
    }

    /**
    * Collect and get rates for storefront
    *
    * @param RateRequest $request
    * @return DataObject|bool|null
    * @api
    */
    public function collectRates(\Magento\Quote\Model\Quote\Address\RateRequest $request)
    {
        if (!$this->isActive()) {
            return false;
        }

        if (empty($this->getConfigData('shipper')) || empty($this->getConfigData('key'))) {
            return false;
        }

        // Exclusion de CP
        $excludedPostcodes = $this->_scopeConfig->getValue('carriers/' . $this->_code . '/excluded_postcodes', \Magento\Store\Model\ScopeInterface::SCOPE_STORE);

        if (!empty($excludedPostcodes)) {
            $excludedPostcodesArray = array_map('trim', explode(',', $excludedPostcodes));
            if (in_array($request->getDestPostcode(), $excludedPostcodesArray)) {
                return false; // No se devuelve tarifa si el CP está excluido
            }
        } 

        /** @var \Magento\Shipping\Model\Rate\Result $result */
        $result = $this->_rateResultFactory->create();
        
        $method = $this->_rateMethodFactory->create();
        $method->setCarrier($this->getCarrierCode());
        $method->setCarrierTitle($this->getConfigData('title'));
        $method->setMethod($this->getCarrierCode());
        $method->setMethodTitle($this->getConfigData('name'));
        
        try {
            $shippingPrice = $this->checkPricePreferences($request->getDestPostcode());
        } catch (\Exception $e) {
            switch ($e->getMessage()) {
                case 'error_postTarifa':
                    $error = $this->_rateErrorFactory->create();
                    $error->setCarrier($this->_code);
                    $error->setCarrierTitle($this->getConfigData('title'));

                    $message = "No se puede obtener la tarifa.";
                    $error->setErrorMessage(__("Error: {$message}"));

                    return $error;
                break;
                
                case 'error_valorTarifa':
                    return false;
                break;
                
                default:
                    $this->_logger->log(100, "Fatal Error: {$e->getMessage()}");
                    return false;
                break;
            }
        }

        $method->setPrice($shippingPrice);
        $method->setCost($shippingPrice);
        
        $result->append($method);

        return $result;
    }

    /**
     * Revisa las opciones personalizadas para precios de Urbano.
     *
     * @param int|string $zipcode
     * @return \Exception|int|float
     */
    protected function checkPricePreferences($zipcode)
    {
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();

        if (filter_var($this->getConfigData('freeShipping'), FILTER_VALIDATE_BOOLEAN) === true) {
            $totals = $objectManager->get('\Magento\Checkout\Model\Cart')->getQuote()->getTotals();
            $subtotal = $totals["subtotal"]->getValue();

            if ($this->getConfigData('freeShippingFrom') <= $subtotal) {
                return 0.00;
            }
        }

        $this->_enabled_custom_price = false;
        $this->_shipping_price = -1;

        $provinces_prices = [
            [
                // Ciudad Autonoma de Buenos Aires
                'zip_codes' => $this->getConfigData('cp_ciudad_autonoma_de_buenos_aires'),
                'price_in_settings' => 'txt_shipping_price_ciudad_autonoma_de_buenos_aires'
            ],
            [
                // Catamarca
                'zip_codes' => $this->getConfigData('cp_catamarca'),
                'price_in_settings' => 'txt_shipping_price_catamarca'
            ],
            [
                // Cordoba
                'zip_codes' => $this->getConfigData('cp_cordoba'),
                'price_in_settings' => 'txt_shipping_price_cordoba'
            ],
            [
                // Corrientes
                'zip_codes' => $this->getConfigData('cp_corrientes'),
                'price_in_settings' => 'txt_shipping_price_corrientes'
            ],
            [
                // Chaco
                'zip_codes' => $this->getConfigData('cp_chaco'),
                'price_in_settings' => 'txt_shipping_price_chaco'
            ],
            [
                // Chubut
                'zip_codes' => $this->getConfigData('cp_chubut'),
                'price_in_settings' => 'txt_shipping_price_chubut'
            ],
            [
                // Entre Rios
                'zip_codes' => $this->getConfigData('cp_entre_rios'),
                'price_in_settings' => 'txt_shipping_price_entre_rios'
            ],
            [
                // Formosa
                'zip_codes' => $this->getConfigData('cp_formosa'),
                'price_in_settings' => 'txt_shipping_price_formosa'
            ],
            [
                // Jujuy
                'zip_codes' => $this->getConfigData('cp_jujuy'),
                'price_in_settings' => 'txt_shipping_price_jujuy'
            ],
            [
                // La Pampa
                'zip_codes' => $this->getConfigData('cp_la_pampa'),
                'price_in_settings' => 'txt_shipping_price_la_pampa'
            ],
            [
                // La Rioja
                'zip_codes' => $this->getConfigData('cp_la_rioja'),
                'price_in_settings' => 'txt_shipping_price_la_rioja'
            ],
            [
                // Mendoza
                'zip_codes' => $this->getConfigData('cp_mendoza'),
                'price_in_settings' => 'txt_shipping_price_mendoza'
            ],
            [
                // Misiones
                'zip_codes' => $this->getConfigData('cp_misiones'),
                'price_in_settings' => 'txt_shipping_price_misiones'
            ],
            [
                // Neuquen
                'zip_codes' => $this->getConfigData('cp_neuquen'),
                'price_in_settings' => 'txt_shipping_price_neuquen'
            ],
            [
                // Rio Negro
                'zip_codes' => $this->getConfigData('cp_rio_negro'),
                'price_in_settings' => 'txt_shipping_price_rio_negro'
            ],
            [
                // Salta
                'zip_codes' => $this->getConfigData('cp_salta'),
                'price_in_settings' => 'txt_shipping_price_salta'
            ],
            [
                // San Juan
                'zip_codes' => $this->getConfigData('cp_san_juan'),
                'price_in_settings' => 'txt_shipping_price_san_juan'
            ],
            [
                // San Luis
                'zip_codes' => $this->getConfigData('cp_san_luis'),
                'price_in_settings' => 'txt_shipping_price_san_luis'
            ],
            [
                // Santa Cruz
                'zip_codes' => $this->getConfigData('cp_santa_cruz'),
                'price_in_settings' => 'txt_shipping_price_santa_cruz'
            ],
            [
                // Santa Fe
                'zip_codes' => $this->getConfigData('cp_santa_fe'),
                'price_in_settings' => 'txt_shipping_price_santa_fe'
            ],
            [
                // Santiago Del Estero
                'zip_codes' => $this->getConfigData('cp_santiago_del_estero'),
                'price_in_settings' => 'txt_shipping_price_santiago_del_estero'
            ],
            [
                // Tucuman
                'zip_codes' => $this->getConfigData('cp_tucuman'),
                'price_in_settings' => 'txt_shipping_price_tucuman'
            ],
            [
                // Tierra Del Fuego
                'zip_codes' => $this->getConfigData('cp_tierra_del_fuego'),
                'price_in_settings' => 'txt_shipping_price_tierra_del_fuego'
            ],
            [
                // Buenos Aires
                'zip_codes' => $this->getConfigData('cp_buenos_aires'),
                'price_in_settings' => 'txt_shipping_price_buenos_aires'
            ],
        ];

        for ($i=0; $i < count($provinces_prices); $i++) {
            if ($this->_enabled_custom_price === true) {
                break;
            }

            $this->checkPriceByProvince(
                $provinces_prices[$i]['zip_codes'],
                $zipcode,
                $provinces_prices[$i]['price_in_settings']
            );
        }
    
        if ($this->_enabled_custom_price === true && $this->_shipping_price >= 0 && $this->_shipping_price != '') {
            return $this->_shipping_price;
        }

        $request = $this->_urbano->postTarifa($zipcode);
        if ($request['success'] == "false") {
            throw new \Exception("error_postTarifa");
        }

        if (!isset($request["valorTarifa"])) {
            throw new \Exception("error_valorTarifa");
        }

        if ( is_numeric($this->getConfigData('impuesto')) && $this->getConfigData('impuesto') > 0  )
            $tarifa = ( ($request["valorTarifa"] * $this->getConfigData('impuesto') ) / 100) + $request["valorTarifa"];
        else
            $tarifa = $request["valorTarifa"];

        return $tarifa;
    }

    /**
     * Verifica si existe un precio personalizado para el codigo postal.
     *
     * A partir de un conjunto de codigos postales, verifica si el codigo postal
     * solicitado se encuentra ahi y busca el precio definido por el
     * administrador.
     *
     * @param string $zipcodes  Codigos postales separados por coma (,) y
     *                          sin espacios para crear un array.
     *
     * @param int|string $zipcode Codigo postal enviado por el cliente.
     *
     * @param string $config_key_price  Nombre del campo de configuracion que
     *                                  guarda el precio personalizado.
     * @return void
     */
    protected function checkPriceByProvince($zipcodes, $zipcode, $config_key_price)
    {
        $zipcodes = explode(',', $zipcodes);
        if (in_array($zipcode, $zipcodes)) {
            $this->_shipping_price = $this->getConfigData($config_key_price);
            $this->_enabled_custom_price = true;
        }
    }

    /**
     * Generacion de datos para la ventana de consulta de tracking.
     *
     * @param string $trackingNumber
     * @return \Magento\Shipping\Model\Tracking\Result\Status
     */
    public function getTrackingInfo($trackingNumber)
    {
        $tracking = $this->_trackStatusFactory->create();

        $endpoints = $this->_urbano->getEndPoints();
        $params = http_build_query([
            "shi_codigo" => $this->getConfigData('shipper'),
            "cli_codigo" => $trackingNumber
        ]);
        $url = "{$this->getConfigData('gateway')}{$endpoints['tracking']}?{$params}";

        $tracking->setData([
            'carrier' => $this->_code,
            'carrier_title' => $this->getConfigData('title'),
            'tracking' => $trackingNumber,
            'url' => $url,
        ]);

        return $tracking;
    }
}
