<?php

namespace Urbano\Shipping\Controller\Adminhtml\Order;

use Magento\Framework\App\Request\Http as HttpRequest;

class EnviarUrbano extends \Magento\Backend\App\Action
{
    /**
     * Authorization level of a basic admin session
     *
     * @see _isAllowed()
     */
    const ADMIN_RESOURCE = 'Magento_Sales::sales_order';

    /** @var \Magento\Sales\Model\ResourceModel\Order\CollectionFactory */
    protected $_orderCollectionFactory;

    /** @var \Psr\Log\LoggerInterface */
    protected $_logger;

    /** @var \Urbano\Shipping\Helper\Urbano */
    protected $_urbano;

    /** @var \Urbano\Shipping\Helper\Config */
    protected $_config;

    /**
     * @param \Urbano\Moduleshipping\Helper\Urbano $urbano
     * @param \Urbano\Shipping\Helper\Config $config
     * @param \Psr\Log\LoggerInterface $logger
     * @param \Magento\Backend\App\Action\Context $context
     * @param \Magento\Sales\Model\ResourceModel\Order\CollectionFactory $orderCollectionFactory
     */
    public function __construct(
        \Urbano\Shipping\Helper\Urbano $urbano,
        \Urbano\Shipping\Helper\Config $config,
        \Psr\Log\LoggerInterface $logger,
        \Magento\Backend\App\Action\Context $context,
        \Magento\Sales\Model\ResourceModel\Order\CollectionFactory $orderCollectionFactory
    ) {
        $this->_orderCollectionFactory = $orderCollectionFactory;
        $this->_logger = $logger;
        $this->_urbano = $urbano;
        $this->_config = $config->getConfig();

        parent::__construct($context);
    }

    public function execute()
    {
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();

        if (!$this->_config['active'] || empty($this->_config['shipper']) || empty($this->_config['key'])) {
            $this->getMessageManager()->addErrorMessage(
                __('No se ha realizado ningun cambio. Por favor revise la configuracion del modulo de Urbano.')
            );
            return $this->_redirect('sales/order/index');
        }

        $ordersShipped = [
            "success" => 0,
            "not_for_urbano" => [],
            "not_valid_for_ship" => [],
            "urbano3_error" => [],
            "tracking_number_apply" => []
        ];

        /** @var HttpRequest $request */
        $request = $this->getRequest();

        $selected = $request->getPost('selected', []);
        $excluded = $request->getPost('excluded', []);

        $orderCollection = $this->_orderCollectionFactory->create();
        if (!empty($selected)) {
            $orderCollection->addFieldToFilter('entity_id', ['in' => $selected]);
        }
        if (!empty($excluded)) {
            $orderCollection->addFieldToFilter('entity_id', ['neq' => $excluded]);
        }

        try {
            /** @var \Magento\Sales\Model\Order $order */
            foreach ($orderCollection as $order) { 
                $orderId = $order->getIncrementId();

                if ($order->getShippingMethod() != "urbano_shipping_urbano_shipping" && $order->getShippingMethod() != "urbanoppc_shipping_urbanoppc_shipping") {
                    $ordersShipped["not_for_urbano"][] = $orderId;
                    continue;
                }

                if (!$order->canShip()) {
                    $ordersShipped["not_valid_for_ship"][] = $orderId;
                    continue;
                }

                $convertOrder = $objectManager->create('Magento\Sales\Model\Convert\Order');
                $shipment = $convertOrder->toShipment($order);

                $body = $this->_urbano->prepareBodyPostCargaClientes($order);
        
                /** @var \Magento\Sales\Model\Order\Item $orderItem */
                foreach ($order->getAllItems() as $orderItem) {
                    if (! $orderItem->getQtyToShip() || $orderItem->getIsVirtual()) {
                        continue;
                    }
        
                    $qtyShipped = $orderItem->getQtyToShip();
                    $shipmentItem = $convertOrder->itemToShipmentItem($orderItem)->setQty($qtyShipped);
                    $shipment->addItem($shipmentItem);
                }

                $shipment->register();

                $request = $this->_urbano->postCargaClientes($body);
                if ($request['success'] == "false") {
                    $ordersShipped["urbano3_error"][] = $orderId;
                    continue;
                }

                $data = [
                    'carrier_code' => 'urbano_shipping',
                    'title' => 'Tracking ID',
                    'number' => $orderId,
                ];

                $shipment->getOrder()->setIsInProcess(true);
        
                try {
                    $track = $objectManager->create('Magento\Sales\Model\Order\Shipment\TrackFactory')->create()->addData($data);
                    $shipment->addTrack($track)->save();
                    $shipment->save();
                    $shipment->getOrder()->save();
            
                    $objectManager->create('Magento\Shipping\Model\ShipmentNotifier')->notify($shipment);
            
                    $shipment->save();
                } catch (\Exception $e) {
                    $ordersShipped["tracking_number_apply"][] = $orderId;
                    continue;
                }
                
                $ordersShipped["success"]++;
            }
        } catch (\Exception $e) {
            $message = "Ha ocurrido un error: {$e->getMessage()}";
            $this->getMessageManager()->addErrorMessage(__($message));
        } finally {
            if ($ordersShipped["success"] > 0) {
                if ($ordersShipped["success"] == 1) {
                    $this->getMessageManager()->addSuccessMessage(__('Se ha enviado 1 orden.'));
                } elseif ($ordersShipped["success"] > 1) {
                    $this->getMessageManager()->addSuccessMessage(__('Se han enviado %1 ordenes.', $ordersShipped["success"]));
                }
            }

            if ($ordersShipped["not_for_urbano"]) {
                $orderIds = implode(", ", $ordersShipped["not_for_urbano"]);
                $this->getMessageManager()->addErrorMessage(
                    __("Las ordenes: {$orderIds}, no tienen a Urbano como metodo de envio.")
                );
            }

            if ($ordersShipped["not_valid_for_ship"]) {
                $orderIds = implode(", ", $ordersShipped["not_valid_for_ship"]);
                $this->getMessageManager()->addErrorMessage(
                    __("Las ordenes: {$orderIds}, no son validas para envio.")
                );
            }

            if ($ordersShipped["urbano3_error"]) {
                $orderIds = implode(", ", $ordersShipped["urbano3_error"]);
                $this->getMessageManager()->addErrorMessage(
                    __("Las ordenes: {$orderIds}, no devolvieron ningun numero de tracking. No se aplico ningun cambio.")
                );
            }

            if ($ordersShipped["tracking_number_apply"]) {
                $orderIds = implode(", ", $ordersShipped["tracking_number_apply"]);
                $this->getMessageManager()->addErrorMessage(
                    __("Las ordenes: {$orderIds}, presentaron errores durante la aplicacion del numero de tracking.")
                );
            }
        }

        return $this->_redirect('sales/order/index');
    }
}
