<?php

namespace Urbanoppc\Shipping\Helper;

class Urbano extends \Magento\Framework\App\Helper\AbstractHelper
{
    /**
     * Numero de Shipper
     *
     * @var int|string
     */
    protected $shipper;

    protected $_cart;

    /**
     * Clave de acceso del shipper
     *
     * @var string
     */
    protected $password;
    
    /**
     * Ruta base de API Urbanoppc
     *
     * @var string
     */
    protected $base_url;

    /**
     * EndPoints disponibles para consumo
     *
     * @var array
     */
    protected $endpoints;

    /** @var \Urbanoppc\Shipping\Helper\Config */
    protected $_config;

    /** @var \Urbano\Shipping\Helper\Config */
    protected $_config2;

    /** @var \Magento\Framework\HTTP\ZendClientFactory */
    protected $_httpClient;

    /** @var \Psr\Log\LoggerInterface */
    protected $_logger;

    /**
     * @param \Psr\Log\LoggerInterface $logger
     * @param \Magento\Framework\HTTP\ZendClientFactory $httpClient
     * @param \Urbanoppc\Moduleshipping\Helper\Config $config
     * @param \Urbano\Moduleshipping\Helper\Config $config2
     */
    public function __construct(
        \Psr\Log\LoggerInterface $logger,
        \Magento\Framework\HTTP\ZendClientFactory $httpClient,
        \Magento\Checkout\Model\Cart $cartModel,
        \Urbanoppc\Shipping\Helper\Config $config,
        \Urbano\Shipping\Helper\Config $config2


    ) {
        $this->_config = $config->getConfig();
        $this->_config2 = $config2->getConfig();

        $this->shipper = $this->_config['shipper'];
        $this->password = $this->_config['key'];
        $this->base_url = $this->_config2['gateway'];
        $this->_cart = $cartModel;
        $this->endpoints = [
            "tarifa" => "/consulta_tarifa_rest/",
            "carga_clientes" => "/cargaCliente/",
            "tracking" => "/cespecifica/",
            "tracking_masivo" => "/tracking/consultaTrackingMasivo/",
            "impresion_etiqueta" => "impresionEtiqueta/"
        ];

        $this->_httpClient = $httpClient;
        $this->_logger = $logger;
    }


    /**
     * Devuelve los EndPoints disponibles.
     *
     * @return array
     */
    public function getEndPoints()
    {
        return $this->endpoints;
    }

    /**
     * Tarifa de Urbanoppc
     *
     * Obtiene mediante API la tarifa de Urbanoppc de acuerdo al
     * codigo postal enviado.
     *
     * @param int|string $zipcode
     * @return array
     */
    public function postTarifa($zipcode)
    {
        try {
            $headers = [
                "content-type" => "application/json",
            ];

            // ... (Code for calculating weight, dimensions, etc.) ...

            $params = [
                // ... (Request parameters) ...
            ];

            $uri = "{$this->base_url}{$this->endpoints['tarifa']}";

            // Initialize cURL
            $ch = curl_init($uri);

            // Set cURL options
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($params));
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

            // Send the request and get the response
            $response = curl_exec($ch);

            // Check for errors
            $responseCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            if ($responseCode != 200) {
                // Handle errors
                return [
                    "success" => "false",
                    "code" => $responseCode,
                    "message" => "API request failed",
                ];
            }

            $requestBody = json_decode($response, true);

            if ($requestBody['codError'] != 0) {
                // Handle API errors
                return [
                    "success" => "false",
                    "code" => $responseCode,
                    "data" => $requestBody,
                ];
            }

            // Success
            $response = array_merge(["success" => "true", "code" => $responseCode], $requestBody);

            return $response;

        } catch (\Exception $e) {
            return [
                "success" => false,
                "code" => $e->getCode(),
                "message" => $e->getMessage(),
            ];
        } finally {
            // Close the cURL handle
            curl_close($ch);
        }
    }


    /**
     * Genera una nueva guia.
     *
     * Envia el/los productos a Urbanoppc para generar una nueva guia.
     *
     * @param array $body
     * @return object
     */
    public function postCargaClientes($body)
    {
        try {
            $headers = [
            "content-type" => "application/json",
            ];

            $uri = "{$this->base_url}{$this->endpoints['carga_clientes']}";

            // Initialize cURL
            $ch = curl_init($uri);

            // Set cURL options
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($body));
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

            // Send the request and get the response
            $response = curl_exec($ch);
            $responseCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

            // Close the cURL handle
            curl_close($ch);

            $requestBody = json_decode($response, true);

            // Check for errors
            if ($responseCode != 200 || $requestBody['resultado'] == "error") {
            $response = [
                "success" => "false",
                "code" => $responseCode,
            ] + $requestBody;

            $this->_logger->log(100, print_r($response, true));
            return $response;
            }

            $response = [
            "success" => "true",
            "code" => $responseCode,
            ] + $requestBody;

            $this->_logger->log(100, print_r($body, true));
            $this->_logger->log(100, print_r($response, true));

            return $response;

        } catch (\Exception $e) {
            return [
            "success" => false,
            "code" => $e->getCode(),
            "message" => $e->getMessage(),
            ];
        }
    }


    public function postTracking($guias)
    {
        try {
            $headers = [
            "content-type" => "application/json",
            ];

            $body = [
            "autentificacion" => [
                "shipper" => $this->shipper,
                "password" => $this->password,
            ],
            "datos" => [],
            ];

            // Build the "datos" array using a loop
            foreach ($guias as $guia) {
            $body['datos'][] = [
                "codigoSeguimiento" => $guia['codigoSeguimiento'],
                "codigoAlternativo" => $guia['codigoAlternativo'],
            ];
            }

            $uri = "{$this->base_url}{$this->endpoints['tracking_masivo']}";

            // Initialize cURL
            $ch = curl_init($uri);

            // Set cURL options
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($body));
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

            // Send the request and get the response
            $response = curl_exec($ch);
            $responseCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

            // Close the cURL handle
            curl_close($ch);

            $requestBody = json_decode($response, true);

            // Check for errors
            if ($responseCode != 200 || $requestBody['resultado'] == "error") {
            $response = [
                "success" => "false",
                "code" => $responseCode,
            ] + $requestBody;

            $this->_logger->log(100, print_r($response, true));
            return $response;
            }

            $response = [
            "success" => "true",
            "code" => $responseCode,
            ] + $requestBody;

            $this->_logger->log(100, print_r($response, true));

            return $response;

        } catch (\Exception $e) {
            return [
            "success" => false,
            "code" => $e->getCode(),
            "message" => $e->getMessage(),
            ];
        }
    }


    /**
     * Hace la peticion a la API Rest de Urbanoppc
     *
     * @param string $url
     * @param string $method
     * @param object $jsonBody
     * @param array $headers
     * @return object
     */
    private function caller($url, $method, $jsonBody, $headers = [])
    {
        $options = [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
        ];

        if ($method == "POST") {
            $options[CURLOPT_POST] = true;
            $options[CURLOPT_POSTFIELDS] = json_encode($jsonBody);
        }

        if (!empty($headers)) {
            $options[CURLOPT_HTTPHEADER] = [];
            foreach ($headers as $key => $value) {
            $options[CURLOPT_HTTPHEADER][] = "$key: $value";
            }
        }

        $ch = curl_init();
        curl_setopt_array($ch, $options);
        $response = curl_exec($ch);
        curl_close($ch);

        // Assuming you want a decoded JSON response
        return json_decode($response, true);
    }

    /**
     * Prepara el cuerpo para la solicitud a CargaClientes de Urbanoppc
     *
     * @param \Magento\Sales\Model\Order $order
     *
     * @return array
     */
    public function prepareBodyPostCargaClientes($order)
    {
        $address = $order->getShippingAddress();
        $uea_sd_cod_poe = null;
        $observaciones = "Codigo POE";
        $servicio = "T";
        $body = [
            'autentificacion' => [
                'shipper' => $this->_config['shipper'],
                'password' => $this->_config['key'],
                'uea_sd_shipper_id' => $this->_config['shipper'],
                'uea_sd_shipper_pass' => $this->_config['key'],
                'uea_sd_cod_poe' => $this->_config['poe'],
            ],
            'destinatario' => [
                'tipoDocumento' => '',
                'numeroDocumento' => '',
                'nombre' => $address->getName(),
                'email' => [
                    $address->getEmail(),
                ],
                'telefono' => $address->getTelephone(),
                'celular' => '',
            ],
            'autorizado' => [
                [
                    'tipoDocumento' => '',
                    'numeroDocumento' => '',
                    'nombre' => '',
                    'email' => [],
                    'telefono' => '',
                    'celular' => '',
                ],
            ],
            'domicilio' => [
                'direccion' => $address->getData('street'),
                'altura' => '',
                'piso' => '',
                'departamento' => '',
                'codigoPostal' => $address->getPostcode(),
                'localidad' => $address->getCity(),
                'provincia' => $address->getRegion(),
                'latitud' => '',
                'longitud' => '',
                'telefono' => [],
            ],
            'domicilioAlternativo' => [
                'direccion' => '',
                'altura' => '',
                'piso' => '',
                'departamento' => '',
                'codigoPostal' => '',
                'localidad' => '',
                'provincia' => '',
                'latitud' => '',
                'longitud' => '',
                'telefono' => [],
            ],
            'datoNumerico' => '',
            'codigoSeguimiento' => $order->getIncrementId(),
            'codigoAlternativo' => '',
            'modeloSms' => $this->_config['modeloSms'],
            'modeloEmail' => $this->_config['modeloEmail'],
            'servicio' => 'E', // Entrega
            'marcaDeAgua' => '',
            'remito' => '',
            'observaciones' => $observaciones,
        ];

        if (
            !empty($this->_config['activeContraReembolso']) &&
            $order->getPayment()->getMethod() == $this->_config['activeContraReembolso']
        ) {
            $body['servicio'] = 'B';
        }

        $body['productos'] = $this->addProductsToPostBody($order, $body['servicio']);

        return $body;
    }


    /**
     * Cree en contenido de Productos para la solicitud a Urbanoppc.
     *
     * @param \Magento\Sales\Model\Order $order
     * @param string $servicio
     *
     * @return array
     */
    protected function addProductsToPostBody($order, $servicio)
    {
        $productos = [];

        switch ($servicio) {
            case 'E':
                /** @var \Magento\Sales\Model\Order\Item $orderItem */
                foreach ($order->getAllItems() as $orderItem) {
                    if (!$orderItem->getQtyToShip() || $orderItem->getIsVirtual()) {
                        continue;
                    }

                    $productos[] = [
                        'largo' => 0,
                        'alto' => 0,
                        'ancho' => 0,
                        'peso' => $orderItem->getWeight(),
                        'valor' => 0,
                        'valorContrareembolso' => 0,
                        'cantidad' => 1,  // Pensado siempre como cross
                        'sku' => '', // Si es warehousing, en caso de crossdocking va vacio
                        'descripcionProducto' => $order->getIncrementId(),
                    ];
                }
            break;

            case 'B':
                /** @var \Magento\Sales\Model\Order\Item $orderItem */
                foreach ($order->getAllItems() as $orderItem) {
                    if (!$orderItem->getQtyToShip() || $orderItem->getIsVirtual()) {
                        continue;
                    }

                    $productos[] = [
                        'largo' => 0,
                        'alto' => 0,
                        'ancho' => 0,
                        'peso' => $orderItem->getWeight(),
                        'valor' => 0,
                        'valorContrareembolso' => $orderItem->getPrice(),
                        'cantidad' => 1,  // Pensado siempre como cross
                        'sku' => '', // Si es warehousing, en caso de crossdocking va vacio
                        'descripcionProducto' => $order->getIncrementId(),
                    ];
                }
            break;
        }

        return $productos;
    }
}
